<?php
// send_message.php
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type");
header("Content-Type: application/json");

// Handle preflight request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require_once __DIR__ . '/../config/database.php';

try {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data) {
        throw new Exception('Invalid JSON data');
    }
    
    // Validate required fields
    $requiredFields = ['trade_id', 'user_id', 'message', 'receiver_id'];
    foreach ($requiredFields as $field) {
        if (!isset($data[$field]) || empty($data[$field])) {
            throw new Exception("Missing required field: $field");
        }
    }
    
    $tradeId = (int)$data['trade_id'];
    $userId = (int)$data['user_id'];
    $message = trim($data['message']);
    $receiver_id = (int)$data['receiver_id'];
    $senderusername = $data['senderusername'] ?? 'User';
    $receiver_username = $data['receiverusername'] ?? 'User';
    
    if (empty($message)) {
        throw new Exception('Message cannot be empty');
    }
    
    // Verify trade exists
    $stmt = $pdo->prepare("SELECT id, buyer_id, seller_id FROM trades WHERE id = ?");
    $stmt->execute([$tradeId]);
    $trade = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$trade) {
        throw new Exception('Trade not found');
    }
    
    // Verify user is part of this trade
    if ($userId != $trade['buyer_id'] && $userId != $trade['seller_id']) {
        throw new Exception('User is not part of this trade');
    }
    
    // Verify receiver is the other party in the trade
    $otherPartyId = ($userId == $trade['buyer_id']) ? $trade['seller_id'] : $trade['buyer_id'];
    if ($receiver_id != $otherPartyId) {
        // Auto-correct receiver_id to the other party
        $receiver_id = $otherPartyId;
        
        // Try to get receiver username from database
        $stmt = $pdo->prepare("SELECT username FROM users WHERE id = ?");
        $stmt->execute([$receiver_id]);
        $receiver = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($receiver) {
            $receiver_username = $receiver['username'];
        }
    }
    
    // Insert message
    $stmt = $pdo->prepare("
        INSERT INTO chat (trade_id, sender_id, receiver_id, sender_username, receiver_username, message, created_at) 
        VALUES (?, ?, ?, ?, ?, ?, NOW())
    ");
    $stmt->execute([$tradeId, $userId, $receiver_id, $senderusername, $receiver_username, $message]);
    
    $messageId = $pdo->lastInsertId();
    
    // Send real-time message via WebSocket
    require_once __DIR__ . '/../crispapi/notification_helper.php';
    sendTradeNotification($tradeId, [
        'type' => 'message',
        'message_id' => $messageId,
        'trade_id' => $tradeId,
        'sender_id' => $userId,
        'receiver_id' => $receiver_id,
        'sender_username' => $senderusername,
        'receiver_username' => $receiver_username,
        'message' => $message,
        'created_at' => date('Y-m-d H:i:s'),
    ]);
    
    // Send push notification to receiver
    sendNotification($receiver_id, $receiver_username, [
        'type' => 'trade_message',
        'title' => 'New Message',
        'message' => "{$senderusername}: {$message}",
        'data' => [
            'tradeId' => $tradeId,
            'sender_id' => $userId,
            'sender_username' => $senderusername,
        ],
        'timestamp' => date('c'),
    ]);
    
    echo json_encode([
        'success' => true,
        'message_id' => $messageId,
        'created_at' => date('Y-m-d H:i:s')
    ]);
    
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Database error: ' . $e->getMessage()
    ]);
    error_log("Send message error: " . $e->getMessage());
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
?>