<?php
// run_migration.php
// Helper script to run the migration programmatically
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json");

require_once __DIR__ . '/../config/database.php';

try {
    // Read the simple migration file
    $migrationFile = __DIR__ . '/migrate_trades_simple.sql';
    
    if (!file_exists($migrationFile)) {
        http_response_code(404);
        echo json_encode(['success' => false, 'error' => 'Migration file not found']);
        exit;
    }
    
    $sql = file_get_contents($migrationFile);
    
    // Split by semicolons and execute each statement
    $statements = array_filter(
        array_map('trim', explode(';', $sql)),
        function($stmt) {
            return !empty($stmt) && !preg_match('/^--/', $stmt);
        }
    );
    
    $executed = [];
    $errors = [];
    
    foreach ($statements as $statement) {
        try {
            // Skip comments and empty lines
            if (empty(trim($statement)) || strpos(trim($statement), '--') === 0) {
                continue;
            }
            
            $pdo->exec($statement);
            $executed[] = $statement;
        } catch (PDOException $e) {
            // If it's a "duplicate column" error, that's okay
            if (strpos($e->getMessage(), 'Duplicate column') !== false || 
                strpos($e->getMessage(), 'already exists') !== false) {
                // Column already exists, skip
                continue;
            }
            $errors[] = [
                'statement' => substr($statement, 0, 100) . '...',
                'error' => $e->getMessage()
            ];
        }
    }
    
    if (count($errors) > 0) {
        echo json_encode([
            'success' => false,
            'error' => 'Some migration statements failed',
            'executed' => count($executed),
            'errors' => $errors
        ]);
    } else {
        echo json_encode([
            'success' => true,
            'message' => 'Migration completed successfully',
            'statements_executed' => count($executed)
        ]);
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
?>

