<?php
// report_trade.php
// Allows users to report/dispute a trade
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json");

// Enable error logging
ini_set('display_errors', 0);
ini_set('log_errors', 1);
error_reporting(E_ALL);

// Log function
function logError($message, $data = null) {
    $logMessage = date("Y-m-d H:i:s") . " - [report_trade] " . $message;
    if ($data !== null) {
        $logMessage .= " | Data: " . json_encode($data);
    }
    $logMessage .= PHP_EOL;
    error_log($logMessage);
    file_put_contents(__DIR__ . '/report_trade_errors.log', $logMessage, FILE_APPEND);
}

logError("Request received", $_SERVER['REQUEST_METHOD']);

require_once __DIR__ . '/../config/database.php';

// Check if notification_helper exists
$notificationHelperPath = __DIR__ . '/notification_helper.php';
if (!file_exists($notificationHelperPath)) {
    logError("Notification helper not found", ['path' => $notificationHelperPath]);
    // Continue without notification helper
    function sendNotification($userId, $title, $message, $data = []) {
        logError("sendNotification called but helper not loaded", ['userId' => $userId, 'title' => $title]);
        return false;
    }
} else {
    require_once $notificationHelperPath;
}

try {
    $rawInput = file_get_contents("php://input");
    logError("Raw input received", ['length' => strlen($rawInput), 'content' => substr($rawInput, 0, 200)]);
    
    $data = json_decode($rawInput, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        logError("JSON decode error", ['error' => json_last_error_msg(), 'input' => $rawInput]);
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Invalid JSON: ' . json_last_error_msg()]);
        exit;
    }
    
    logError("JSON decoded successfully", $data);
    
    $tradeId = $data['trade_id'] ?? 0;
    $userId = $data['user_id'] ?? 0;
    $reason = $data['reason'] ?? '';
    $description = $data['description'] ?? '';
    
    logError("Parsed parameters", ['tradeId' => $tradeId, 'userId' => $userId, 'reason' => $reason]);
    
    if (!$tradeId || !$userId || !$reason) {
        logError("Missing required parameters", ['tradeId' => $tradeId, 'userId' => $userId, 'reason' => $reason]);
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Trade ID, User ID, and reason are required']);
        exit;
    }
    
    // Check database connection
    if (!isset($pdo)) {
        logError("PDO not initialized");
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => 'Database connection not initialized']);
        exit;
    }
    
    // Get trade details
    try {
        logError("Fetching trade details", ['tradeId' => $tradeId]);
        $stmt = $pdo->prepare("SELECT * FROM trades WHERE id = ?");
        $stmt->execute([$tradeId]);
        $trade = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$trade) {
            logError("Trade not found", ['tradeId' => $tradeId]);
            http_response_code(404);
            echo json_encode(['success' => false, 'error' => 'Trade not found']);
            exit;
        }
        
        logError("Trade found", ['tradeId' => $tradeId, 'buyerId' => $trade['buyer_id'] ?? 'N/A', 'sellerId' => $trade['seller_id'] ?? 'N/A', 'status' => $trade['status'] ?? 'N/A']);
    } catch (PDOException $e) {
        logError("Error fetching trade", ['error' => $e->getMessage(), 'code' => $e->getCode()]);
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => 'Database error fetching trade: ' . $e->getMessage()]);
        exit;
    }
    
    // Check if user is part of this trade
    if ($userId != $trade['buyer_id'] && $userId != $trade['seller_id']) {
        logError("User not authorized", ['userId' => $userId, 'buyerId' => $trade['buyer_id'], 'sellerId' => $trade['seller_id']]);
        http_response_code(403);
        echo json_encode(['success' => false, 'error' => 'You are not authorized to report this trade']);
        exit;
    }
    
    // Check if already reported (handle case where column might not exist)
    if (isset($trade['reported_by']) && $trade['reported_by']) {
        logError("Trade already reported", ['reportedBy' => $trade['reported_by']]);
        echo json_encode([
            'success' => true,
            'message' => 'This trade has already been reported',
            'already_reported' => true
        ]);
        exit;
    }
    
    // Update trade with dispute information
    // Check if columns exist first, if not, use a simpler update
    $disputeReason = $reason . ($description ? ': ' . $description : '');
    logError("Updating trade with dispute", ['disputeReason' => $disputeReason, 'userId' => $userId]);
    
    // Try to update with all columns, catch error if columns don't exist
    try {
        logError("Attempting full update with all columns");
        $updateStmt = $pdo->prepare("
            UPDATE trades 
            SET status = 'disputed',
                dispute_reason = ?,
                reported_by = ?,
                reported_at = NOW()
            WHERE id = ?
        ");
        $updateStmt->execute([$disputeReason, $userId, $tradeId]);
        logError("Full update successful");
    } catch (PDOException $e) {
        logError("Full update failed, trying simple update", ['error' => $e->getMessage(), 'code' => $e->getCode()]);
        // If columns don't exist, try updating just status
        // This allows the feature to work even if migration hasn't run
        try {
            logError("Attempting simple status update");
            $updateStmt = $pdo->prepare("
                UPDATE trades 
                SET status = 'disputed'
                WHERE id = ?
            ");
            $updateStmt->execute([$tradeId]);
            logError("Simple update successful");
        } catch (PDOException $e2) {
            logError("Simple update also failed", ['error' => $e2->getMessage(), 'code' => $e2->getCode()]);
            http_response_code(500);
            echo json_encode([
                'success' => false, 
                'error' => 'Failed to update trade status. Please run database migration first.',
                'details' => $e2->getMessage()
            ]);
            exit;
        }
    }
    
    // Add system message to chat
    try {
        logError("Adding system message to chat");
        $otherPartyId = $userId == $trade['buyer_id'] ? $trade['seller_id'] : $trade['buyer_id'];
        logError("Other party ID", ['otherPartyId' => $otherPartyId]);
        
        $userStmt = $pdo->prepare("SELECT username FROM users WHERE id = ?");
        $userStmt->execute([$userId]);
        $userData = $userStmt->fetch(PDO::FETCH_ASSOC);
        $username = $userData['username'] ?? 'User';
        logError("User data fetched", ['username' => $username]);
        
        $msgStmt = $pdo->prepare("
            INSERT INTO chat (trade_id, sender_id, receiver_id, message, created_at)
            VALUES (?, ?, ?, ?, NOW())
        ");
        $message = "⚠️ {$username} has reported this trade. Reason: {$reason}" . ($description ? ". Details: {$description}" : "");
        logError("Executing chat insert", ['tradeId' => $tradeId, 'senderId' => $userId, 'receiverId' => $otherPartyId]);
        
        $msgStmt->execute([
            $tradeId,
            $userId,
            $otherPartyId,
            $message
        ]);
        
        logError("Chat message inserted successfully");
    } catch (PDOException $e) {
        logError("Error inserting chat message", ['error' => $e->getMessage(), 'code' => $e->getCode()]);
        // Continue anyway, don't fail the whole request
    }
    
    // Notify the other party
    try {
        logError("Sending notification to other party", ['otherPartyId' => $otherPartyId]);
        sendNotification($otherPartyId, 'Trade Reported', 'This trade has been reported and is under review.', [
            'type' => 'trade_reported',
            'trade_id' => $tradeId
        ]);
        logError("Notification sent");
    } catch (Exception $e) {
        logError("Error sending notification", ['error' => $e->getMessage()]);
        // Continue anyway
    }
    
    // TODO: Notify all admins about the dispute
    // For now, auto-add admin if available
    try {
        logError("Looking for admin to auto-join");
        $adminStmt = $pdo->prepare("SELECT id FROM users WHERE role = 'admin' OR role = 'Admin' LIMIT 1");
        $adminStmt->execute();
        $admin = $adminStmt->fetch(PDO::FETCH_ASSOC);
        
        if ($admin) {
            logError("Admin found", ['adminId' => $admin['id']]);
            // Try to update admin columns, catch error if they don't exist
            try {
                $adminUpdateStmt = $pdo->prepare("
                    UPDATE trades 
                    SET admin_id = ?, 
                        admin_joined_at = NOW() 
                    WHERE id = ?
                ");
                $adminUpdateStmt->execute([$admin['id'], $tradeId]);
                logError("Admin auto-joined successfully");
            } catch (PDOException $e) {
                logError("Admin columns may not exist", ['error' => $e->getMessage(), 'code' => $e->getCode()]);
                // If admin columns don't exist, just log it
            }
            
            // Notify both parties that admin joined
            try {
                sendNotification($trade['buyer_id'], 'Admin Joined', 'An admin has automatically joined to resolve the dispute.', [
                    'type' => 'admin_joined',
                    'trade_id' => $tradeId
                ]);
                
                sendNotification($trade['seller_id'], 'Admin Joined', 'An admin has automatically joined to resolve the dispute.', [
                    'type' => 'admin_joined',
                    'trade_id' => $tradeId
                ]);
                logError("Admin join notifications sent");
            } catch (Exception $e) {
                logError("Error sending admin join notifications", ['error' => $e->getMessage()]);
            }
        } else {
            logError("No admin found to auto-join");
        }
    } catch (PDOException $e) {
        logError("Error finding admin", ['error' => $e->getMessage()]);
        // Continue anyway
    }
    
    logError("Report completed successfully");
    echo json_encode([
        'success' => true,
        'message' => 'Trade reported successfully. An admin will review it shortly.',
        'disputed' => true
    ]);
} catch (PDOException $e) {
    logError("PDO Exception caught", ['error' => $e->getMessage(), 'code' => $e->getCode(), 'file' => $e->getFile(), 'line' => $e->getLine()]);
    http_response_code(500);
    echo json_encode([
        'success' => false, 
        'error' => 'Database error: ' . $e->getMessage(),
        'details' => $e->getMessage()
    ]);
} catch (Exception $e) {
    logError("General Exception caught", ['error' => $e->getMessage(), 'file' => $e->getFile(), 'line' => $e->getLine()]);
    http_response_code(500);
    echo json_encode([
        'success' => false, 
        'error' => $e->getMessage(),
        'details' => $e->getMessage()
    ]);
}
?>

