<?php
require_once __DIR__ . '/../config/database.php';

header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type");
header("Content-Type: application/json");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

try {
    $input = json_decode(file_get_contents('php://input'), true);
    
    // Validate input
    if (empty($input['username']) || empty($input['password'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Email/Username and password are required']);
        exit;
    }

    $username = trim($input['username']);
    $password = $input['password'];

    // Get user from database - check if input is email or username
    $isEmail = filter_var($username, FILTER_VALIDATE_EMAIL);
    $field = $isEmail ? 'email' : 'username';

    $stmt = $pdo->prepare("
        SELECT id, username, email, password, ip, rating, balance, created_at
        FROM users
        WHERE $field = ?
    ");
    $stmt->execute([$username]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$user) {
        http_response_code(401);
        echo json_encode(['error' => 'Invalid email/username or password']);
        exit;
    }

    // Verify password
    if (!password_verify($password, $user['password'])) {
        http_response_code(401);
        echo json_encode(['error' => 'Invalid email/username or password']);
        exit;
    }

    // Generate a simple token (in production, use JWT)
    $token = bin2hex(random_bytes(32));

    // Update last login time
    $stmt = $pdo->prepare("UPDATE users SET last_login = NOW() WHERE id = ?");
    $stmt->execute([$user['id']]);

    // Return user data (without password)
    echo json_encode([
        'success' => true,
        'message' => 'Login successful',
        'user' => [
            'id' => (string)$user['id'],
            'username' => $user['username'],
            'email' => $user['email'],
            'rating' => $user['rating'],
            'balance' => $user['balance']
        ],
        'token' => $token
    ]);

} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Database error: ' . $e->getMessage()]);
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Server error: ' . $e->getMessage()]);
}
?>
