-- Create order book tables for exchange trading
-- This creates the orders table for buy/sell orders and exchange_trades table for completed trades

-- 1. Create orders table for buy/sell orders
CREATE TABLE IF NOT EXISTS `orders` (
  `id` INT(11) NOT NULL AUTO_INCREMENT,
  `user_id` INT(11) NOT NULL,
  `username` VARCHAR(45) NOT NULL,
  `pair` VARCHAR(20) NOT NULL COMMENT 'Trading pair e.g., BTC/USDT',
  `base_coin` VARCHAR(10) NOT NULL COMMENT 'Base coin e.g., BTC',
  `quote_coin` VARCHAR(10) NOT NULL COMMENT 'Quote coin e.g., USDT',
  `order_type` ENUM('buy', 'sell') NOT NULL,
  `order_side` ENUM('limit', 'market') NOT NULL DEFAULT 'limit',
  `price` DECIMAL(20, 8) NOT NULL COMMENT 'Price per unit in quote currency',
  `amount` DECIMAL(20, 8) NOT NULL COMMENT 'Amount of base coin',
  `filled_amount` DECIMAL(20, 8) NOT NULL DEFAULT 0.00000000 COMMENT 'Amount already filled',
  `remaining_amount` DECIMAL(20, 8) NOT NULL COMMENT 'Amount remaining to fill',
  `total` DECIMAL(20, 8) NOT NULL COMMENT 'Total value in quote currency (price * amount)',
  `status` ENUM('pending', 'partially_filled', 'filled', 'cancelled') NOT NULL DEFAULT 'pending',
  `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  INDEX `idx_pair_type_status` (`pair`, `order_type`, `status`),
  INDEX `idx_user_status` (`user_id`, `status`),
  INDEX `idx_price` (`pair`, `order_type`, `price`),
  INDEX `idx_created` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 2. Create exchange_trades table for completed trades
CREATE TABLE IF NOT EXISTS `exchange_trades` (
  `id` INT(11) NOT NULL AUTO_INCREMENT,
  `trade_id` VARCHAR(50) NOT NULL COMMENT 'Unique trade identifier',
  `buy_order_id` INT(11) DEFAULT NULL COMMENT 'Reference to buy order',
  `sell_order_id` INT(11) DEFAULT NULL COMMENT 'Reference to sell order',
  `buyer_id` INT(11) NOT NULL,
  `buyer_username` VARCHAR(45) NOT NULL,
  `seller_id` INT(11) NOT NULL,
  `seller_username` VARCHAR(45) NOT NULL,
  `pair` VARCHAR(20) NOT NULL COMMENT 'Trading pair e.g., BTC/USDT',
  `base_coin` VARCHAR(10) NOT NULL COMMENT 'Base coin e.g., BTC',
  `quote_coin` VARCHAR(10) NOT NULL COMMENT 'Quote coin e.g., USDT',
  `price` DECIMAL(20, 8) NOT NULL COMMENT 'Execution price',
  `amount` DECIMAL(20, 8) NOT NULL COMMENT 'Amount of base coin traded',
  `total` DECIMAL(20, 8) NOT NULL COMMENT 'Total value in quote currency',
  `fee` DECIMAL(20, 8) NOT NULL DEFAULT 0.00000000 COMMENT 'Trading fee',
  `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `unique_trade_id` (`trade_id`),
  INDEX `idx_pair_created` (`pair`, `created_at`),
  INDEX `idx_buyer` (`buyer_id`, `created_at`),
  INDEX `idx_seller` (`seller_id`, `created_at`),
  INDEX `idx_created` (`created_at`),
  FOREIGN KEY (`buy_order_id`) REFERENCES `orders`(`id`) ON DELETE SET NULL,
  FOREIGN KEY (`sell_order_id`) REFERENCES `orders`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

