import React, { createContext, useContext, useState, useEffect, useRef } from 'react';
import { router } from 'expo-router';
import { supabase } from '@/lib/supabase';

export interface User {
  id: string;
  username: string;
  displayName: string;
  profileImage?: string;
  bio?: string;
  following: number;
  followers: number;
  joinDate: string;
}

interface AuthContextType {
  isAuthenticated: boolean;
  user: User | null;
  isLoading: boolean;
  login: (username: string, password: string) => Promise<void>;
  logout: () => Promise<void>;
  register: (username: string, email: string, password: string) => Promise<void>;
}

export const AuthContext = createContext<AuthContextType>({
  isAuthenticated: false,
  user: null,
  isLoading: true,
  login: async () => {},
  logout: async () => {},
  register: async () => {},
});

export const AuthProvider: React.FC<{children: React.ReactNode}> = ({ children }) => {
  const [isAuthenticated, setIsAuthenticated] = useState<boolean>(false);
  const [user, setUser] = useState<User | null>(null);
  const [isLoading, setIsLoading] = useState<boolean>(true);
  const isMounted = useRef(true);

  useEffect(() => {
    checkUser();

    const { data: authListener } = supabase.auth.onAuthStateChange(async (event, session) => {
      if (!isMounted.current) return;

      if (event === 'SIGNED_IN') {
        const user = session?.user;
        if (user) {
          await fetchUserProfile(user.id);
        }
      } else if (event === 'SIGNED_OUT') {
        setIsAuthenticated(false);
        setUser(null);
      }
    });

    return () => {
      isMounted.current = false;
      authListener.subscription.unsubscribe();
    };
  }, []);

  const checkUser = async () => {
    try {
      const { data: { session } } = await supabase.auth.getSession();
      if (session?.user && isMounted.current) {
        await fetchUserProfile(session.user.id);
      }
    } catch (error) {
      console.error('Error checking auth status:', error);
    } finally {
      if (isMounted.current) {
        setIsLoading(false);
      }
    }
  };

  const fetchUserProfile = async (userId: string) => {
    if (!isMounted.current || !userId) return;

    try {
      const { data: profile, error } = await supabase
        .from('users')
        .select('*')
        .eq('id', userId)
        .maybeSingle();

      if (error) throw error;

      if (!profile) {
        console.error('User profile not found:', userId);
        setIsAuthenticated(false);
        setUser(null);
        return;
      }

      if (profile && isMounted.current) {
        setUser({
          id: profile.id,
          username: profile.username,
          displayName: profile.display_name,
          profileImage: profile.profile_image_url,
          bio: profile.bio,
          following: profile.following_count,
          followers: profile.followers_count,
          joinDate: new Date(profile.created_at).toLocaleDateString('en-US', { month: 'long', year: 'numeric' }),
        });
        setIsAuthenticated(true);
      }
    } catch (error) {
      console.error('Error fetching user profile:', error);
      if (isMounted.current) {
        setIsAuthenticated(false);
        setUser(null);
      }
    }
  };

  const login = async (username: string, password: string) => {
    if (!isMounted.current) return;
    setIsLoading(true);

    try {
      const { data: { user }, error } = await supabase.auth.signInWithPassword({
        email: username, // Using username as email since that's how we registered
        password,
      });

      if (error) throw error;

      if (user && isMounted.current) {
        await fetchUserProfile(user.id);
        router.replace('/(tabs)');
      }
    } catch (error) {
      console.error('Login error:', error);
      throw new Error('Login failed. Please check your credentials.');
    } finally {
      if (isMounted.current) {
        setIsLoading(false);
      }
    }
  };

  const logout = async () => {
    if (!isMounted.current) return;

    try {
      await supabase.auth.signOut();
      if (isMounted.current) {
        setIsAuthenticated(false);
        setUser(null);
        router.replace('/(auth)');
      }
    } catch (error) {
      console.error('Logout error:', error);
      throw new Error('Logout failed. Please try again.');
    }
  };

  const register = async (username: string, email: string, password: string) => {
    if (!isMounted.current) return;
    setIsLoading(true);

    try {
      // Check if username already exists
      const { data: existingUser, error: checkError } = await supabase
        .from('users')
        .select('id')
        .eq('username', username)
        .maybeSingle();

      if (checkError) throw checkError;
      
      if (existingUser) {
        throw new Error('Username already taken. Please choose another.');
      }

      const { data: { user }, error: signUpError } = await supabase.auth.signUp({
        email,
        password,
      });

      if (signUpError) throw signUpError;

      if (!user) {
        throw new Error('Failed to create user account.');
      }

      // Create user profile
      const { error: profileError } = await supabase
        .from('users')
        .insert([
          {
            id: user.id,
            username,
            display_name: username,
            created_at: new Date().toISOString(),
          },
        ]);

      if (profileError) {
        throw new Error('Failed to create user profile. Please try again.');
      }

      if (user && isMounted.current) {
        await fetchUserProfile(user.id);
        router.replace('/(tabs)');
      }
    } catch (error) {
      console.error('Registration error:', error);
      throw error instanceof Error ? error : new Error('Registration failed. Please try again.');
    } finally {
      if (isMounted.current) {
        setIsLoading(false);
      }
    }
  };

  return (
    <AuthContext.Provider value={{ 
      isAuthenticated, 
      user, 
      isLoading,
      login,
      logout,
      register
    }}>
      {children}
    </AuthContext.Provider>
  );
};

export const useAuth = () => useContext(AuthContext);