<?php
// admin_resolve_dispute.php
// Allows admin to resolve disputes and release escrow to either party
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json");

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/notification_helper.php';

try {
    $data = json_decode(file_get_contents("php://input"), true);
    $tradeId = $data['trade_id'] ?? 0;
    $adminId = $data['admin_id'] ?? 0;
    $action = $data['action'] ?? ''; // 'release_to_buyer' or 'release_to_seller' or 'refund_both'
    $resolutionNote = $data['resolution_note'] ?? '';
    
    if (!$tradeId || !$adminId || !$action) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Trade ID, Admin ID, and action are required']);
        exit;
    }
    
    // Verify admin
    $adminStmt = $pdo->prepare("SELECT role FROM users WHERE id = ?");
    $adminStmt->execute([$adminId]);
    $admin = $adminStmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$admin || ($admin['role'] !== 'admin' && $admin['role'] !== 'Admin')) {
        http_response_code(403);
        echo json_encode(['success' => false, 'error' => 'Admin privileges required']);
        exit;
    }
    
    // Get trade details
    $stmt = $pdo->prepare("SELECT * FROM trades WHERE id = ?");
    $stmt->execute([$tradeId]);
    $trade = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$trade) {
        http_response_code(404);
        echo json_encode(['success' => false, 'error' => 'Trade not found']);
        exit;
    }
    
    if ($trade['status'] !== 'disputed') {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Trade is not in disputed status']);
        exit;
    }
    
    // Get escrow balance (you'll need to implement this based on your escrow system)
    // For now, we'll use the trade amount as escrow
    $escrowAmount = $trade['amount'] ?? 0;
    $coin = $trade['coin'] ?? 'USDT';
    
    // Get user balances
    $buyerStmt = $pdo->prepare("SELECT * FROM wallets WHERE user_id = ? AND coin = ?");
    $sellerStmt = $pdo->prepare("SELECT * FROM wallets WHERE user_id = ? AND coin = ?");
    
    $buyerStmt->execute([$trade['buyer_id'], $coin]);
    $sellerStmt->execute([$trade['seller_id'], $coin]);
    
    $buyerWallet = $buyerStmt->fetch(PDO::FETCH_ASSOC);
    $sellerWallet = $sellerStmt->fetch(PDO::FETCH_ASSOC);
    
    $pdo->beginTransaction();
    
    try {
        if ($action === 'release_to_buyer') {
            // Release escrow to buyer (refund)
            if ($buyerWallet) {
                $updateBuyer = $pdo->prepare("UPDATE wallets SET balance = balance + ? WHERE id = ?");
                $updateBuyer->execute([$escrowAmount, $buyerWallet['id']]);
            } else {
                // Create wallet if doesn't exist
                $createBuyer = $pdo->prepare("INSERT INTO wallets (user_id, coin, balance) VALUES (?, ?, ?)");
                $createBuyer->execute([$trade['buyer_id'], $coin, $escrowAmount]);
            }
            
            // Record transaction
            $txStmt = $pdo->prepare("
                INSERT INTO transactions (user_id, type, amount, coin, status, description, created_at)
                VALUES (?, 'p2p', ?, ?, 'Completed', ?, NOW())
            ");
            $txStmt->execute([
                $trade['buyer_id'],
                $escrowAmount,
                $coin,
                "Admin resolution: Escrow refunded. {$resolutionNote}"
            ]);
            
            $recipientId = $trade['buyer_id'];
            $recipientType = 'buyer';
            
        } else if ($action === 'release_to_seller') {
            // Release escrow to seller (complete trade)
            if ($sellerWallet) {
                $updateSeller = $pdo->prepare("UPDATE wallets SET balance = balance + ? WHERE id = ?");
                $updateSeller->execute([$escrowAmount, $sellerWallet['id']]);
            } else {
                $createSeller = $pdo->prepare("INSERT INTO wallets (user_id, coin, balance) VALUES (?, ?, ?)");
                $createSeller->execute([$trade['seller_id'], $coin, $escrowAmount]);
            }
            
            $txStmt = $pdo->prepare("
                INSERT INTO transactions (user_id, type, amount, coin, status, description, created_at)
                VALUES (?, 'p2p', ?, ?, 'Completed', ?, NOW())
            ");
            $txStmt->execute([
                $trade['seller_id'],
                $escrowAmount,
                $coin,
                "Admin resolution: Escrow released. {$resolutionNote}"
            ]);
            
            $recipientId = $trade['seller_id'];
            $recipientType = 'seller';
            
        } else if ($action === 'refund_both') {
            // Split escrow (50/50 or custom split - simplified here)
            $halfAmount = $escrowAmount / 2;
            
            // Refund buyer
            if ($buyerWallet) {
                $updateBuyer = $pdo->prepare("UPDATE wallets SET balance = balance + ? WHERE id = ?");
                $updateBuyer->execute([$halfAmount, $buyerWallet['id']]);
            } else {
                $createBuyer = $pdo->prepare("INSERT INTO wallets (user_id, coin, balance) VALUES (?, ?, ?)");
                $createBuyer->execute([$trade['buyer_id'], $coin, $halfAmount]);
            }
            
            // Refund seller
            if ($sellerWallet) {
                $updateSeller = $pdo->prepare("UPDATE wallets SET balance = balance + ? WHERE id = ?");
                $updateSeller->execute([$halfAmount, $sellerWallet['id']]);
            } else {
                $createSeller = $pdo->prepare("INSERT INTO wallets (user_id, coin, balance) VALUES (?, ?, ?)");
                $createSeller->execute([$trade['seller_id'], $coin, $halfAmount]);
            }
            
            $recipientId = null;
            $recipientType = 'both';
        } else {
            throw new Exception('Invalid action. Must be release_to_buyer, release_to_seller, or refund_both');
        }
        
        // Update trade status
        $updateStmt = $pdo->prepare("
            UPDATE trades 
            SET status = 'completed',
                completed_at = NOW()
            WHERE id = ?
        ");
        $updateStmt->execute([$tradeId]);
        
        // Add system message
        $msgStmt = $pdo->prepare("
            INSERT INTO chat (trade_id, sender_id, receiver_id, message, created_at)
            VALUES (?, ?, ?, ?, NOW())
        ");
        
        $adminStmt = $pdo->prepare("SELECT username FROM users WHERE id = ?");
        $adminStmt->execute([$adminId]);
        $adminData = $adminStmt->fetch(PDO::FETCH_ASSOC);
        $adminUsername = $adminData['username'] ?? 'Admin';
        
        $resolutionMsg = "✅ Admin {$adminUsername} resolved this dispute. ";
        if ($action === 'release_to_buyer') {
            $resolutionMsg .= "Escrow refunded to buyer.";
        } else if ($action === 'release_to_seller') {
            $resolutionMsg .= "Escrow released to seller.";
        } else {
            $resolutionMsg .= "Escrow split between both parties.";
        }
        if ($resolutionNote) {
            $resolutionMsg .= " Note: {$resolutionNote}";
        }
        
        // Message to buyer
        $msgStmt->execute([$tradeId, $adminId, $trade['buyer_id'], $resolutionMsg]);
        // Message to seller
        $msgStmt->execute([$tradeId, $adminId, $trade['seller_id'], $resolutionMsg]);
        
        // Notify parties
        sendNotification($trade['buyer_id'], 'Dispute Resolved', $resolutionMsg, [
            'type' => 'dispute_resolved',
            'trade_id' => $tradeId
        ]);
        
        sendNotification($trade['seller_id'], 'Dispute Resolved', $resolutionMsg, [
            'type' => 'dispute_resolved',
            'trade_id' => $tradeId
        ]);
        
        $pdo->commit();
        
        echo json_encode([
            'success' => true,
            'message' => 'Dispute resolved successfully',
            'action' => $action,
            'recipient' => $recipientType
        ]);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
    
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Database error: ' . $e->getMessage()]);
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
}
?>

